#include "stdafx.h"
#include "Debug.h"
#include "Common.h"
#include "PNM_Config.h"


/*************************************************************************************************
 * @brief This method builds a configuration packet for the IO-Controller.
 * 
 * @param PNM_APPLICATION_T* ptPnmApp  pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt pointer to a CIFX_PACKET structur used for receiving the request
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigIocReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
  TLR_RESULT tResult = TLR_S_OK;

  PNM_APCFG_CFG_PNM_EXT_REQ_T* ptConfigReq = (PNM_APCFG_CFG_PNM_EXT_REQ_T*)ptSendPkt;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                 /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                 /* Source of packet, process queue         */
  ptConfigReq->tHead.ulLen  = sizeof(ptConfigReq->tData);           /* Length of packet data without header    */
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_PNM_EXT_REQ; /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                    /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                    /* extension                               */

  /* fill the subheader contained in every configuration packet */
  ptConfigReq->tData.tSubHead.ulTrCntrId   = 0; /* this is the IO-Controller dataset */
  ptConfigReq->tData.tSubHead.ulTrDevId    = 0; /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrApId     = 0; /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrModId    = 0; /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrSubModId = 0; /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrIdCnt    = 1; /* amount if datasets in this packet */

  /* fill the dataset */
  ptConfigReq->tData.atData[0].ulTrId          = 1;           /* this is the first (and only) dataset of the packet */
  ptConfigReq->tData.atData[0].ulIPAddr        = 0xC0A8Fc7F;  /* 192.168.252.127 */
  ptConfigReq->tData.atData[0].ulNetmask       = 0xFFFFFF00;  /* 255.255.255.0 */
  ptConfigReq->tData.atData[0].ulGateway       = 0;
  ptConfigReq->tData.atData[0].ulIPFlags       = 0;
  ptConfigReq->tData.atData[0].ulSystemFlags   = PNM_APCFG_STARTMODE_APPLICATION; /* application controlled startup */
  ptConfigReq->tData.atData[0].ulWdgTime       = 1000;        /* DPM watchdog */
  ptConfigReq->tData.atData[0].usAlign1        = 0;
  ptConfigReq->tData.atData[0].usAlign2        = 0;
  ptConfigReq->tData.atData[0].usDeviceID      = 0x0203;      /* Hilscher cifX DeviceID for IO-Controller */
  ptConfigReq->tData.atData[0].usVendorID      = 0x011E;      /* Hilscher Profinet VendorID */
  ptConfigReq->tData.atData[0].ulStructVersion = PNM_APCFG_CFG_PNM_STRUCT_VERS_1;
  ptConfigReq->tData.atData[0].usMAUTypePort0  = PNM_APCFG_PNM_MAUTYPE_AUTO; /* enable AutoNegotiation and AutoCrossover */
  ptConfigReq->tData.atData[0].usMAUTypePort1  = PNM_APCFG_PNM_MAUTYPE_AUTO; /* enable AutoNegotiation and AutoCrossover */
  ptConfigReq->tData.atData[0].usNameOfStationLen = strlen("controller-example");
  ptConfigReq->tData.atData[0].usTypeOfStationLen = strlen("cifX RE/PNM");
  memcpy(&ptConfigReq->tData.atData[0].abNameOfStation, "controller-example", strlen("controller-example"));
  memcpy(&ptConfigReq->tData.atData[0].abTypeOfStation, "cifX RE/PNM", strlen("cifX RE/PNM"));

  /* send the configuration packet */
  DEBUG("Sending PNM configuration request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_PNM_EXT_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This method builds a configuration packet for the basic IO-Device parameters.
 * 
 * @param PNM_APPLICATION_T* ptPnmApp  pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt pointer to a CIFX_PACKET structur used for receiving the request
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigIodReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
  TLR_RESULT tResult = TLR_S_OK;
  TLR_UINT   uiCnt;

  PNM_APCFG_CFG_IOD_REQ_T* ptConfigReq = (PNM_APCFG_CFG_IOD_REQ_T*)ptSendPkt;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                 /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                 /* Source of packet, process queue         */
  ptConfigReq->tHead.ulLen  = sizeof(ptConfigReq->tData.tSubHead) + /* Length of packet data without header    */
                              ptPnmApp->tPnm.usIoDeviceCount * sizeof(ptConfigReq->tData.atData);
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_IOD_REQ; /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                    /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                    /* extension                               */


  /* fill the subheader contained in every configuration packet */
  ptConfigReq->tData.tSubHead.ulTrCntrId   = 1; /* this dataset belongs to IO-Controller 1 */
  ptConfigReq->tData.tSubHead.ulTrDevId    = 0; /* this is the IO-Device configuration packet */
  ptConfigReq->tData.tSubHead.ulTrApId     = 0; /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrModId    = 0; /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrSubModId = 0; /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrIdCnt    = ptPnmApp->tPnm.usIoDeviceCount; /* amount of datasets in this packet */

  /* fill the dataset */
  for (uiCnt = 0; uiCnt < ptConfigReq->tData.tSubHead.ulTrIdCnt; uiCnt++)
  {
    ptConfigReq->tData.atData[uiCnt].ulTrId          = uiCnt +1; /* dataset number of this dataset in the packet */

    /* fill in the device specific parameters */
    ptConfigReq->tData.atData[uiCnt].ulFlags         = PNM_APCFG_IOD_FLAG_IOD_ACTIVE; 
    ptConfigReq->tData.atData[uiCnt].usDeviceID      = ptPnmApp->tPnm.patIoDevices[uiCnt].usDeviceId;
    ptConfigReq->tData.atData[uiCnt].usVendorID      = ptPnmApp->tPnm.patIoDevices[uiCnt].usVendorId;
    ptConfigReq->tData.atData[uiCnt].usObjUUIDInst   = ptPnmApp->tPnm.patIoDevices[uiCnt].usInstance;
    ptConfigReq->tData.atData[uiCnt].ulIPAddr        = ptPnmApp->tPnm.patIoDevices[uiCnt].ulIpAddr;
    ptConfigReq->tData.atData[uiCnt].ulNetworkMask   = ptPnmApp->tPnm.patIoDevices[uiCnt].ulNetmask;
    ptConfigReq->tData.atData[uiCnt].ulGatewayAddr   = ptPnmApp->tPnm.patIoDevices[uiCnt].ulGateway;

    ptConfigReq->tData.atData[uiCnt].ulArUuidData1   = ptPnmApp->tPnm.patIoDevices[uiCnt].ulArUuid1;
    ptConfigReq->tData.atData[uiCnt].usArUuidData2   = ptPnmApp->tPnm.patIoDevices[uiCnt].usArUuid2;
    ptConfigReq->tData.atData[uiCnt].usArUuidData3   = ptPnmApp->tPnm.patIoDevices[uiCnt].usArUuid3;
    memcpy(&ptConfigReq->tData.atData[uiCnt].abArUuidData4, &ptPnmApp->tPnm.patIoDevices[uiCnt].abArUuid4, sizeof(ptConfigReq->tData.atData[uiCnt].abArUuidData4));
    ptConfigReq->tData.atData[uiCnt].usNameOfStationLen = strlen((const char*)ptPnmApp->tPnm.patIoDevices[uiCnt].abNameOfStation);
    memcpy(&ptConfigReq->tData.atData[uiCnt].abNameOfStation, &ptPnmApp->tPnm.patIoDevices[uiCnt].abNameOfStation, strlen((const char*)ptPnmApp->tPnm.patIoDevices[uiCnt].abNameOfStation));

    /* now fill in the static parameters */
    ptConfigReq->tData.atData[uiCnt].usARType           = PNIO_API_AR_TYPE_SINGLE;
    ptConfigReq->tData.atData[uiCnt].ulARProp           = PNIO_API_AR_PROP_STATE_PRIMARY | PNIO_API_AR_PROP_SUPERVISOR_NONE | PNIO_API_AR_PROP_SINGLE_AR;
    ptConfigReq->tData.atData[uiCnt].usAlarmCRType      = PNIO_API_ALCR_TYPE_ALARM;
    ptConfigReq->tData.atData[uiCnt].ulAlarmCRProp      = PNIO_API_ALCR_PROP_PRIO_DEFAULT;
    ptConfigReq->tData.atData[uiCnt].usAlarmCRVLANID    = 0;                                 /* no special VLAN ID is used for Profinet */
    ptConfigReq->tData.atData[uiCnt].ulIPFlags          = 0;
    ptConfigReq->tData.atData[uiCnt].usRTATimeoutFact   = 100;                               /* use default value 100ms as RTA Timeout base */
    ptConfigReq->tData.atData[uiCnt].usRTARetries       = PNIO_API_RTA_RETRIES_MIN;
    ptConfigReq->tData.atData[uiCnt].usAlign1           = 0;
    ptConfigReq->tData.atData[uiCnt].usAlign2           = 0;
    ptConfigReq->tData.atData[uiCnt].usAlign3           = 0;
    ptConfigReq->tData.atData[uiCnt].usAlign4           = 0;
    ptConfigReq->tData.atData[uiCnt].usAlign5           = 0;
  }

  /* send the configuration packet */
  DEBUG("Sending IOD configuration request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_IOD_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }
  
  return tResult;
}

/*************************************************************************************************
 * @brief This method builds a configuration packet for the IOCRs of a single IO-Device.
 * 
 * @param PNM_APPLICATION_T* ptPnmApp pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt pointer to a CIFX_PACKET structur used for receiving the request
 * @param TLR_UINT           uiIodCnt is the number of the IO-Device in the application parameter array
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigIocrReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt, TLR_UINT uiIodCnt)
{
  TLR_RESULT tResult = TLR_S_OK;
  PNM_IO_DEVICE_PARAMETERS_T* ptDevPrm = (PNM_IO_DEVICE_PARAMETERS_T*)&ptPnmApp->tPnm.patIoDevices[uiIodCnt];
  PNM_SUBMODULE_T* ptSubm;
  TLR_UINT uiInputLength, uiOutputLength, uiCnt;

  PNM_APCFG_CFG_IOCR_REQ_T* ptConfigReq = (PNM_APCFG_CFG_IOCR_REQ_T*)ptSendPkt;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                 /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                 /* Source of packet, process queue         */
  ptConfigReq->tHead.ulLen  = sizeof(ptConfigReq->tData.tSubHead) + /* Length of packet data without header    */
                              2 * sizeof(ptConfigReq->tData.atData);
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_IOD_IOCR_REQ; /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                    /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                    /* extension                               */

  /* first calculate the complete input and output length of this IO-Device */
  uiInputLength  = 0;
  uiOutputLength = 0;
  /* iterate over all submodules of the IO-Device and calculate the frame size for InputCR and OutputCR */
  /* the IOPS and IOCS elements have to be counted as well! */
  for (uiCnt = 0; uiCnt < ptDevPrm->usSubmodCnt; uiCnt++)
  {
    ptSubm = (PNM_SUBMODULE_T*)&ptDevPrm->patSubm[uiCnt];
    if (0 == ptSubm->usInputLength && 0 == ptSubm->usOutputLength)
    {
      /* only the IOPS and IOCS have to be counted */
      uiInputLength  += 1; /* IOPS of dataless submodule */
      uiOutputLength += 1; /* IOCS of dataless submodule */
    }
    if (0 != ptSubm->usOutputLength)
    {
      uiOutputLength += ptSubm->usOutputLength;
      uiOutputLength += 1; /* count the IOPS of the submodule */
      uiInputLength  += 1; /* IOCS */
    }
    if (0 != ptSubm->usInputLength)
    {
      uiInputLength  += ptSubm->usInputLength;
      uiInputLength  += 1; /* count the IOPS of the submodule */
      uiOutputLength += 1; /* IOCS */
    }
  }

  /* fill the subheader contained in every configuration packet */
  ptConfigReq->tData.tSubHead.ulTrCntrId   = 1;           /* this dataset belongs to IO-Controller 1 */
  ptConfigReq->tData.tSubHead.ulTrDevId    = uiIodCnt +1; /* the IO-Device this dataset belongs to */
  ptConfigReq->tData.tSubHead.ulTrApId     = 0;           /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrModId    = 0;           /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrSubModId = 0;           /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrIdCnt    = 2;           /* amount of datasets in this packet (PNM only supports exactly 2 IOCRs per IO-Device) */

  /* fill the dataset */
  
  /* InputCR */
  ptConfigReq->tData.atData[0].ulTrId          = 1; /* dataset count of this dataset in the packet */
  ptConfigReq->tData.atData[0].usType          = PNIO_API_IOCR_TYPE_INPUT;
  ptConfigReq->tData.atData[0].usVLANID        = 0; /* no special VLAN ID is used for Profinet */
  ptConfigReq->tData.atData[0].ulProp          = PNIO_API_IOCR_PROP_RTCLASS_DATA1;
  memset(&ptConfigReq->tData.atData[0].abMcastMACAddr, 0, sizeof(ptConfigReq->tData.atData[0].abMcastMACAddr));
  ptConfigReq->tData.atData[0].usDataLen       = uiInputLength;
  ptConfigReq->tData.atData[0].usSendClockFact = ptDevPrm->usSendClock;
  ptConfigReq->tData.atData[0].usReductRatio   = ptDevPrm->usReductRatio;
  ptConfigReq->tData.atData[0].usPhase         = 1;
  ptConfigReq->tData.atData[0].usSequ          = 0; /* currently unused */
  ptConfigReq->tData.atData[0].ulFrameSendOffs = PNIO_API_FRAME_SEND_OFFSET_DEFAULT;
  ptConfigReq->tData.atData[0].usWatchdogFact  = PNIO_API_CYCLIC_WATCHDOG_DEFAULT;
  ptConfigReq->tData.atData[0].usDataHoldFact  = PNIO_API_CYCLIC_DATAHOLD_DEFAULT;

  /* OutputCR */
  ptConfigReq->tData.atData[1].ulTrId          = 2; /* dataset count of this dataset in the packet */
  ptConfigReq->tData.atData[1].usType          = PNIO_API_IOCR_TYPE_OUTPUT;
  ptConfigReq->tData.atData[1].usVLANID        = 0; /* no special VLAN ID is used for Profinet */
  ptConfigReq->tData.atData[1].ulProp          = PNIO_API_IOCR_PROP_RTCLASS_DATA1;
  memset(&ptConfigReq->tData.atData[1].abMcastMACAddr, 0, sizeof(ptConfigReq->tData.atData[0].abMcastMACAddr));
  ptConfigReq->tData.atData[1].usDataLen       = uiOutputLength;
  ptConfigReq->tData.atData[1].usSendClockFact = ptDevPrm->usSendClock;
  ptConfigReq->tData.atData[1].usReductRatio   = ptDevPrm->usReductRatio;
  ptConfigReq->tData.atData[1].usPhase         = 1;
  ptConfigReq->tData.atData[1].usSequ          = 0; /* currently unused */
  ptConfigReq->tData.atData[1].ulFrameSendOffs = PNIO_API_FRAME_SEND_OFFSET_DEFAULT;
  ptConfigReq->tData.atData[1].usWatchdogFact  = PNIO_API_CYCLIC_WATCHDOG_DEFAULT;
  ptConfigReq->tData.atData[1].usDataHoldFact  = PNIO_API_CYCLIC_DATAHOLD_DEFAULT;

  /* send the configuration packet */
  DEBUG("Sending IOCR configuration request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_IOD_IOCR_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This method builds a configuration packet for the APs of a single IO-Device.
 * 
 * @param PNM_APPLICATION_T* ptPnmApp pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt pointer to a CIFX_PACKET structur used for receiving the request
 * @param TLR_UINT           uiIodCnt is the number of the IO-Device in the application parameter array
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigApReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt, TLR_UINT uiIodCnt)
{
  TLR_RESULT tResult = TLR_S_OK;
  PNM_IO_DEVICE_PARAMETERS_T* ptDevPrm = (PNM_IO_DEVICE_PARAMETERS_T*)&ptPnmApp->tPnm.patIoDevices[uiIodCnt];
  TLR_UINT   uiCnt;
  TLR_UINT32 ulCurrentApi = 0xFFFFFFFF; /* set to an invalid initialization value */

  PNM_APCFG_CFG_AP_REQ_T* ptConfigReq = (PNM_APCFG_CFG_AP_REQ_T*)ptSendPkt;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                 /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                 /* Source of packet, process queue         */
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_IOD_AP_REQ;  /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                    /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                    /* extension                               */


  /* fill the subheader contained in every configuration packet */
  ptConfigReq->tData.tSubHead.ulTrCntrId   = 1;           /* this dataset belongs to IO-Controller 1 */
  ptConfigReq->tData.tSubHead.ulTrDevId    = uiIodCnt +1; /* the IO-Device this dataset belongs to */
  ptConfigReq->tData.tSubHead.ulTrApId     = 0;           /* this is the AP configuration */
  ptConfigReq->tData.tSubHead.ulTrModId    = 0;           /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrSubModId = 0;           /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrIdCnt    = 0;           /* amount of datasets in this packet (increased inside the function !) */

  for (uiCnt = 0; uiCnt < ptDevPrm->usSubmodCnt; uiCnt++)
  {
    if (ulCurrentApi != ptDevPrm->patSubm[uiCnt].ulApi)
    {
      /* add this AP to the packet */
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].ulAPI  = ptDevPrm->patSubm[uiCnt].ulApi;
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].ulTrId = ptConfigReq->tData.tSubHead.ulTrIdCnt +1;
      ptConfigReq->tData.tSubHead.ulTrIdCnt++;
      ulCurrentApi = ptDevPrm->patSubm[uiCnt].ulApi;
    }
  }

  /* calculate the length of the packet */
  ptConfigReq->tHead.ulLen  = sizeof(ptConfigReq->tData.tSubHead) +
                              ptConfigReq->tData.tSubHead.ulTrIdCnt * sizeof(ptConfigReq->tData.atData);

  /* send the configuration packet */
  DEBUG("Sending AP configuration request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_IOD_AP_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This method builds a configuration packet for the Modules of an API of a single IO-Device.
 *        This method is NOT capable of handling multiple APs!
 * 
 * @param PNM_APPLICATION_T* ptPnmApp  pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt pointer to a CIFX_PACKET structur used for receiving the request
 * @param TLR_UINT           uiIodCnt  is the number of the IO-Device in the application parameter array
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigModulesReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt, TLR_UINT uiIodCnt)
{
  TLR_RESULT tResult = TLR_S_OK;
  PNM_IO_DEVICE_PARAMETERS_T* ptDevPrm = (PNM_IO_DEVICE_PARAMETERS_T*)&ptPnmApp->tPnm.patIoDevices[uiIodCnt];
  TLR_UINT   uiCnt;
  TLR_UINT16 usCurrentSlot = 0xFFFF; /* set to an invalid initialization value */

  PNM_APCFG_CFG_MODULE_REQ_T* ptConfigReq = (PNM_APCFG_CFG_MODULE_REQ_T*)ptSendPkt;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                    /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                    /* Source of packet, process queue         */
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_IOD_MODULE_REQ; /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                       /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                       /* extension                               */


  /* fill the subheader contained in every configuration packet */
  ptConfigReq->tData.tSubHead.ulTrCntrId   = 1;           /* this dataset belongs to IO-Controller 1 */
  ptConfigReq->tData.tSubHead.ulTrDevId    = uiIodCnt +1; /* the IO-Device this dataset belongs to */
  ptConfigReq->tData.tSubHead.ulTrApId     = 1;           /* this dataset belongs to the only existing AP */
  ptConfigReq->tData.tSubHead.ulTrModId    = 0;           /* this is the module configuration */
  ptConfigReq->tData.tSubHead.ulTrSubModId = 0;           /* irrelevant for this packet */
  ptConfigReq->tData.tSubHead.ulTrIdCnt    = 0;           /* amount of datasets in this packet (increased inside the function !) */

  for (uiCnt = 0; uiCnt < ptDevPrm->usSubmodCnt; uiCnt++)
  {
    if (usCurrentSlot != ptDevPrm->patSubm[uiCnt].usSlot)
    {
      /* add this module to the packet */
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].ulModuleID   = ptDevPrm->patSubm[uiCnt].ulModuleId;
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].usSlotNumber = ptDevPrm->patSubm[uiCnt].usSlot;
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].usModuleProp = 0; /* currently unused */
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].ulTrId       = ptConfigReq->tData.tSubHead.ulTrIdCnt +1;
      ptConfigReq->tData.tSubHead.ulTrIdCnt++;
      usCurrentSlot = ptDevPrm->patSubm[uiCnt].usSlot;
    }
  }

  /* calculate the length of the packet */
  ptConfigReq->tHead.ulLen  = sizeof(ptConfigReq->tData.tSubHead) +
                              ptConfigReq->tData.tSubHead.ulTrIdCnt * sizeof(ptConfigReq->tData.atData);

  /* send the configuration packet */
  DEBUG("Sending Module configuration request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_IOD_MODULE_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }
  
  return tResult;
}

/*************************************************************************************************
 * @brief This method builds a configuration packet for the Submodules of a module of a single IO-Device.
 * 
 * @param PNM_APPLICATION_T* ptPnmApp  pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt pointer to a CIFX_PACKET structur used for receiving the request
 * @param TLR_UINT           uiIodCnt  is the number of the IO-Device in the application parameter array
 * @param TLR_UINT           uiModId   is the number (counter value) of the module this submodules belongs to
 * @param TLR_UINT16         usSlot    is the Slot of the module this submodules belong to
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigSubmodulesReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt, TLR_UINT uiIodCnt, TLR_UINT uiModId, TLR_UINT16 usSlot)
{
  TLR_RESULT tResult = TLR_S_OK;
  PNM_IO_DEVICE_PARAMETERS_T* ptDevPrm = (PNM_IO_DEVICE_PARAMETERS_T*)&ptPnmApp->tPnm.patIoDevices[uiIodCnt];
  TLR_UINT   uiCnt;

  PNM_APCFG_CFG_SUBMODULE_REQ_T* ptConfigReq = (PNM_APCFG_CFG_SUBMODULE_REQ_T*)ptSendPkt;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                       /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                       /* Source of packet, process queue         */
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_IOD_SUBMODULE_REQ; /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                          /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                          /* extension                               */


  /* fill the subheader contained in every configuration packet */
  ptConfigReq->tData.tSubHead.ulTrCntrId   = 1;           /* this dataset belongs to IO-Controller 1 */
  ptConfigReq->tData.tSubHead.ulTrDevId    = uiIodCnt +1; /* the IO-Device this dataset belongs to */
  ptConfigReq->tData.tSubHead.ulTrApId     = 1;           /* this dataset belongs to the only existing AP */
  ptConfigReq->tData.tSubHead.ulTrModId    = uiModId +1;  /* this dataset belongs to this module identifier */
  ptConfigReq->tData.tSubHead.ulTrSubModId = 0;           /* this is the submodule configuration */
  ptConfigReq->tData.tSubHead.ulTrIdCnt    = 0;           /* amount of datasets in this packet (increased inside the function !) */

  for (uiCnt = 0; uiCnt < ptDevPrm->usSubmodCnt; uiCnt++)
  {
    if (usSlot == ptDevPrm->patSubm[uiCnt].usSlot)
    {
      /* add this submodule to the packet */
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].ulSubmoduleID   = ptDevPrm->patSubm[uiCnt].ulSubmodId;
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].usSubslotNumber = ptDevPrm->patSubm[uiCnt].usSubslot;
      /* set the submodule properties according to the data of submodule */
      if (0 == ptDevPrm->patSubm[uiCnt].usInputLength && 0 == ptDevPrm->patSubm[uiCnt].usOutputLength)
      {
        ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].usSubmoduleProp = PNIO_API_SUBM_PROP_TYPE_NONE;
      }
      else if (0 != ptDevPrm->patSubm[uiCnt].usInputLength && 0 != ptDevPrm->patSubm[uiCnt].usOutputLength)
      {
        ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].usSubmoduleProp = PNIO_API_SUBM_PROP_TYPE_BOTH;
      }
      else if (0 != ptDevPrm->patSubm[uiCnt].usInputLength)
      {
        ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].usSubmoduleProp = PNIO_API_SUBM_PROP_TYPE_INPUT;
      }
      else
      {
        ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].usSubmoduleProp = PNIO_API_SUBM_PROP_TYPE_OUTPUT;
      }
      ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt].ulTrId          = ptConfigReq->tData.tSubHead.ulTrIdCnt +1;
      ptConfigReq->tData.tSubHead.ulTrIdCnt++;
    }
  }

  /* calculate the length of the packet */
  ptConfigReq->tHead.ulLen  = sizeof(ptConfigReq->tData.tSubHead) +
                              ptConfigReq->tData.tSubHead.ulTrIdCnt * sizeof(ptConfigReq->tData.atData);

  /* send the configuration packet */
  DEBUG("Sending Submodule configuration request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_IOD_SUBMODULE_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This method builds a configuration packet for the SubmoduleDescription of a submodule of a single IO-Device.
 * 
 * @param PNM_APPLICATION_T* ptPnmApp    pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt   pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt   pointer to a CIFX_PACKET structur used for receiving the request
 * @param TLR_UINT           uiIodCnt    is the number of the IO-Device in the application parameter array
 * @param TLR_UINT           uiModId     is the number (counter value) of the module this submodules belong to
 * @param TLR_UINT           uiSubmodId  is the number (counter value) of the submodule this SubmoduleDescription belongs to
 * @param PNM_SUBMODULE_T*   ptSubm      pointer to the submodule in the application parameter array
 * @param TLR_UINT*          puiFrameOffsetInputCR   is the current InputCR-FrameOffset  (read and written by the method!)
 * @param TLR_UINT*          puiFrameOffsetOutputCR  is the current OutputCR-FrameOffset (read and written by the method!)
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigSubmoduleDescriptionsReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt, TLR_UINT uiIodCnt, TLR_UINT uiModId, TLR_UINT uiSubmodId, 
                                                   PNM_SUBMODULE_T* ptSubm, TLR_UINT* puiFrameOffsetInputCR, TLR_UINT* puiFrameOffsetOutputCR)
{
  TLR_RESULT tResult = TLR_S_OK;
  PNM_IO_DEVICE_PARAMETERS_T* ptDevPrm = (PNM_IO_DEVICE_PARAMETERS_T*)&ptPnmApp->tPnm.patIoDevices[uiIodCnt];

  PNM_APCFG_CFG_SUBMDESCR_REQ_T* ptConfigReq = (PNM_APCFG_CFG_SUBMDESCR_REQ_T*)ptSendPkt;
  PNM_APCFG_CFG_SUBMDESCR_DATA_T* ptData;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                        /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                        /* Source of packet, process queue         */
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_IOD_SUBMDESCR_REQ; /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                           /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                           /* extension                               */


  /* fill the subheader contained in every configuration packet */
  ptConfigReq->tData.tSubHead.ulTrCntrId   = 1;             /* this dataset belongs to IO-Controller 1*/
  ptConfigReq->tData.tSubHead.ulTrDevId    = uiIodCnt +1;   /* the IO-Device this dataset belongs to */
  ptConfigReq->tData.tSubHead.ulTrApId     = 1;             /* this dataset belongs to the only existing AP */
  ptConfigReq->tData.tSubHead.ulTrModId    = uiModId +1;    /* this dataset belongs to this module identifier */
  ptConfigReq->tData.tSubHead.ulTrSubModId = uiSubmodId +1; /* this dataset belongs to this submodule identifier */
  ptConfigReq->tData.tSubHead.ulTrIdCnt    = 0;             /* amount of datasets in this packet (increased inside the function !) */

  /* submodules with input data or without any IO-data are input submodules */
  if (0 != ptSubm->usInputLength || ((0 == ptSubm->usInputLength && 0 == ptSubm->usOutputLength)))
  {
    /* add dataset for InputCR */
    ptData = (PNM_APCFG_CFG_SUBMDESCR_DATA_T*)&ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt];
    ptData->bIOCSLen        = 1; /* the IOCS has a fixed length of 1 byte */
    ptData->bIOPSLen        = 1; /* the IOPS has a fixed length of 1 byte */
    ptData->ulDPM_Offset    = ptSubm->usDPMOffsIn;
    ptData->ulIO_Block      = 0; /* currently unused */
    ptData->ulIOCRIdCons    = 1; /* this ID must match the TreeID of the InputCR of this IO-Device */
    ptData->ulIOCRIdProd    = 2; /* this ID must match the TreeID of the OutputCR of this IO-Device */
    ptData->ulSignla_Attrib = 0; /* currently unused */
    ptData->ulTrId          = ptConfigReq->tData.tSubHead.ulTrIdCnt + 1;
    ptData->usAlign1        = 0;
    ptData->usDataDescr     = PNIO_API_SUBMDESCR_DATA_DESCR_INPUT;

    /* Frame offset configuration */
    ptData->usFrameOffs     = *puiFrameOffsetInputCR;
    ptData->usIOCSFrameOffs = *puiFrameOffsetOutputCR;
    ptData->usSubmDataLen   = ptSubm->usInputLength;

    /* adjust the variables to handle frame offsets */
    *puiFrameOffsetInputCR  += ptData->usSubmDataLen + 1; /* +1 because of IOPS */
    *puiFrameOffsetOutputCR += 1; /* IOCS */

    /* increase the counter */
    ptConfigReq->tData.tSubHead.ulTrIdCnt++;
  }

  if (0 != ptSubm->usOutputLength)
  {
    /* add dataset for OutputCR */
    ptData = (PNM_APCFG_CFG_SUBMDESCR_DATA_T*)&ptConfigReq->tData.atData[ptConfigReq->tData.tSubHead.ulTrIdCnt];
    ptData->bIOCSLen        = 1; /* the IOCS has a fixed length of 1 byte */
    ptData->bIOPSLen        = 1; /* the IOPS has a fixed length of 1 byte */
    ptData->ulDPM_Offset    = ptSubm->usDPMOffsIn;
    ptData->ulIO_Block      = 0; /* currently unused */
    ptData->ulIOCRIdCons    = 1; /* this ID must match the TreeID of the InputCR of this IO-Device */
    ptData->ulIOCRIdProd    = 2; /* this ID must match the TreeID of the OutputCR of this IO-Device */
    ptData->ulSignla_Attrib = 0; /* currently unused */
    ptData->ulTrId          = ptConfigReq->tData.tSubHead.ulTrIdCnt + 1;
    ptData->usAlign1        = 0;
    ptData->usDataDescr     = PNIO_API_SUBMDESCR_DATA_DESCR_OUTPUT;
    
    /* Frame offset configuration */
    ptData->usFrameOffs     = *puiFrameOffsetOutputCR;
    ptData->usIOCSFrameOffs = *puiFrameOffsetInputCR;
    ptData->usSubmDataLen   = ptSubm->usOutputLength;

    /* adjust the variables to handle frame offsets */
    *puiFrameOffsetOutputCR += ptData->usSubmDataLen + 1; /* +1 because of IOPS */
    *puiFrameOffsetInputCR  += 1; /* IOCS */

    /* increase the counter */
    ptConfigReq->tData.tSubHead.ulTrIdCnt++;
  }

  /* calculate the length of the packet */
  ptConfigReq->tHead.ulLen  = sizeof(ptConfigReq->tData.tSubHead) +
                              ptConfigReq->tData.tSubHead.ulTrIdCnt * sizeof(ptConfigReq->tData.atData);


  /* send the configuration packet */
  DEBUG("Sending SubmoduleDescription configuration request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_IOD_SUBMDESCR_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This method builds a configuration request Download Finished.
 * 
 * @param PNM_APPLICATION_T* ptPnmApp    pointer to the Profinet application parameters
 * @param CIFX_PACKET*       ptSendPkt   pointer to a CIFX_PACKET structur used for sending the request
 * @param CIFX_PACKET*       ptRecvPkt   pointer to a CIFX_PACKET structur used for receiving the request
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigDownloadFinishReq(PNM_APPLICATION_T* ptPnmApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
  TLR_RESULT tResult = TLR_S_OK;
  PNM_APCFG_DWNL_FIN_REQ_T* ptConfigReq = (PNM_APCFG_DWNL_FIN_REQ_T*)ptSendPkt;
  memset(ptConfigReq, 0, sizeof(*ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                  /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc  = 0x00;                                  /* Source of packet, process queue         */
  ptConfigReq->tHead.ulCmd  = PNM_APCTL_CMD_SET_CONFIG_DWNL_FIN_REQ; /* Packet command                          */
  ptConfigReq->tHead.ulSta  = 0;                                     /* Status code of operation                */
  ptConfigReq->tHead.ulExt  = 0;                                     /* extension                               */
  ptConfigReq->tHead.ulLen  = 0;                                     /* Length of packet data without header    */

  /* send the configuration packet */
  DEBUG("Sending Download Finished request...\n");
  tResult = xChannelPutPacket(ptPnmApp->tCommon.hChannel, ptSendPkt, ptPnmApp->tCommon.ulTimeout);
  if (TLR_S_OK == tResult)
  {
    /* get configuration confirmation packet */
    tResult = xChannelGetPacket(ptPnmApp->tCommon.hChannel, sizeof(CIFX_PACKET), ptRecvPkt, ptPnmApp->tCommon.ulTimeout);
    /* check if we got an error within configuration packet */
    tResult = App_CheckPkt(ptRecvPkt, tResult, PNM_APCTL_CMD_SET_CONFIG_DWNL_FIN_CNF, TLR_S_OK);
  }
  else
  {
    DEBUG("Sending packet failed with 0x%08x\n", tResult);
    tResult = RCX_E_QUE_SENDPACKET;
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This method will configure the Profniet IO-Controller firmware by sending the required
 *        packets in the correct order.
 *
 * This method uses the configuration parameters from local application as base for the
 * Profinet configuration.
 * 
 * @param PNM_APPLICATION_T* ptApp       pointer to the application parameters
 * @param CIFX_PACKET*       ptSendPkt    pointer to a CIFX_PACKET structur used for sending packets.
 * @param CIFX_PACKET*       ptRecvPkt    pointer to a CIFX_PACKET structur used for receiving packets.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_ConfigurePnm(APPLICATION_T* ptApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
  TLR_RESULT tResult   = TLR_S_OK;
  TLR_UINT   uiIodCnt;                 /* counter to iterate over array of IO-Devices */
  TLR_UINT   uiSubmCnt;                /* counter to iterate over array of submodules */
  TLR_UINT   uiModIdCnt, uiSubmIdCnt;  /* value of current Tree Identifiers */
  TLR_UINT   uiFrameOffsetInputCr;     /* FrameOffset to be used for next element in InputCR  (used on per-IO-Device basis) */
  TLR_UINT   uiFrameOffsetOutputCr;    /* FrameOffset to be used for next element in OutputCR (used on per-IO-Device basis) */
  TLR_UINT16 usCurrentSlot;            /* the slot of the module currently processed */
  
  /* cast pointer to Profinet application parameters */
  PNM_APPLICATION_T* ptPnmApp = (PNM_APPLICATION_T*)ptApp;


  /* configure basic IO-Controller parameters */
  tResult = App_BuildConfigIocReq(ptPnmApp, ptSendPkt, ptRecvPkt);
  if (TLR_S_OK != tResult)
    goto leave;

  /* configure basic IO-Device parameters */
  tResult = App_BuildConfigIodReq(ptPnmApp, ptSendPkt, ptRecvPkt);
  if (TLR_S_OK != tResult)
    goto leave;

  /* now configure the submodules of each IO-Device */
  /* the sequence is (per IO-Device):
   * IOCR
   * API
   ** all Modules of the API
   *** all Submodules of the current Module
   **** SubmoduleDescription for every submodule 
  */

  /* this code requires the submodules in local resource to be sorted by API, Slot and Subslot */
  for (uiIodCnt=0; uiIodCnt < ptPnmApp->tPnm.usIoDeviceCount; uiIodCnt++)
  {
    /* reset temp. variables */
    uiFrameOffsetInputCr  = 0;
    uiFrameOffsetOutputCr = 0;

    /* configure IOCRs of IO-Device */
    tResult = App_BuildConfigIocrReq(ptPnmApp, ptSendPkt, ptRecvPkt, uiIodCnt);
    if (TLR_S_OK != tResult)
      goto leave;

    /* configure the APIs of IO-Device */
    /* Note: this code is currently only able to handle a single API */
    tResult = App_BuildConfigApReq(ptPnmApp, ptSendPkt, ptRecvPkt, uiIodCnt);
    if (TLR_S_OK != tResult)
      goto leave;

    /* configure the modules of IO-Device */
    tResult = App_BuildConfigModulesReq(ptPnmApp, ptSendPkt, ptRecvPkt, uiIodCnt);
    if (TLR_S_OK != tResult)
      goto leave;

    /* before entering next loop set initial invalid values */
    usCurrentSlot    = 0xFFFF;
    uiModIdCnt       = 0;

    /* add the submodules */
    for (uiSubmCnt = 0; uiSubmCnt < ptPnmApp->tPnm.patIoDevices[uiIodCnt].usSubmodCnt; uiSubmCnt++)
    {
      if (usCurrentSlot != ptPnmApp->tPnm.patIoDevices[uiIodCnt].patSubm[uiSubmCnt].usSlot)
      {
        /* add the submodules of the current module */
        tResult = App_BuildConfigSubmodulesReq(ptPnmApp, ptSendPkt, ptRecvPkt, uiIodCnt, uiModIdCnt, ptPnmApp->tPnm.patIoDevices[uiIodCnt].patSubm[uiSubmCnt].usSlot);
        if (TLR_S_OK != tResult)
          goto leave;
      }

      /* store the current slot number */
      usCurrentSlot = ptPnmApp->tPnm.patIoDevices[uiIodCnt].patSubm[uiSubmCnt].usSlot;

      /* now configure the SubmoduleDescription for all submodules of the current module */
      /* this is done in by an additional loop over the array of submodules */
      /* before entering the loop start with SubmoduleTreeID 0 for every new submodule */
      uiSubmIdCnt = 0;
      while ((usCurrentSlot == ptPnmApp->tPnm.patIoDevices[uiIodCnt].patSubm[uiSubmCnt].usSlot) && 
             (uiSubmCnt < ptPnmApp->tPnm.patIoDevices[uiIodCnt].usSubmodCnt))
      {
        /* add submodule descriptions (each submodule needs its own!) for the current submodule */
        App_BuildConfigSubmoduleDescriptionsReq(ptPnmApp, ptSendPkt, ptRecvPkt, uiIodCnt, uiModIdCnt, uiSubmIdCnt, &ptPnmApp->tPnm.patIoDevices[uiIodCnt].patSubm[uiSubmCnt], &uiFrameOffsetInputCr, &uiFrameOffsetOutputCr);

        /* step to the next submodule and increase the submodule tree identifier */
        uiSubmCnt++;
        uiSubmIdCnt++;
      }

      /* increase the module tree ID */
      uiModIdCnt++;
      /* decrease the submodule counter to not skip a submodule (the while loop increases by 1 to often) */
      uiSubmCnt--;
    }
  }

  /* the configuration sequence is finished, inform the Profinet IO-Controller about this fact */
  tResult = App_BuildConfigDownloadFinishReq(ptPnmApp, ptSendPkt, ptRecvPkt);

leave:
  return tResult;
}

/*************************************************************************************************
 * @brief This method checks a confirmation packet.
 * 
 * @param CIFX_PACKET*       ptRecvPkt    pointer to CIFX_PACKET structur to check
 * @param int32_t            cifXError    the cifX error code generated while getting packet from mailbox
 * @param TLR_UINT32         ulExpCommand the command value expected in the packet
 * @param TLR_UINT32         ulExpStatus  the status value expected in the packet
 *
 * @return TLR_RESULT        Returns TLR_S_OK if all checks match, otherwise it will return an error code.
 *
 */
TLR_RESULT App_CheckPkt(CIFX_PACKET* ptRecvPkt, int32_t cifXError, TLR_UINT32 ulExpCommand, TLR_UINT32 ulExpStatus)
{
  TLR_RESULT tResult = TLR_S_OK;

  if (CIFX_NO_ERROR != cifXError)
  {
    DEBUG("Configuration Confirmation not received. cifXError 0x%08x\n", cifXError);
    tResult = TLR_E_FAIL;
  }
  else if (ulExpCommand != ptRecvPkt->tHeader.ulCmd)
  {
    DEBUG("received unexpected command: 0x%08x\n", ptRecvPkt->tHeader.ulCmd);
    tResult = TLR_E_UNEXPECTED;
  }
  else if (ulExpStatus != ptRecvPkt->tHeader.ulState)
  {
    DEBUG("Configuration packet returned with error code: 0x%x\n", ptRecvPkt->tHeader.ulState );
    tResult = ptRecvPkt->tHeader.ulState;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This function creates all resources for the application. It contains the Profinet configuration.
 * It allocates memory for the application data and returns a pointer to it.
 * App_FreeResources() must be called in order to free resources again.
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_CreateResources(APPLICATION_T** pptApp)
{
  DEBUG("Allocating resources...\n");

  /* return value of function */
  TLR_RESULT tResult      = TLR_S_OK;
  /* helping variable for length calculation */
  TLR_UINT   uiLenMemory = 0;
  /* temp. counter */
  TLR_UINT   uiCnt;

  /* at the end of the function we will return the pointer */
  *pptApp = NULL;

  /* allocate memory for application data */
  PNM_APPLICATION_T* ptPnmApp = (PNM_APPLICATION_T*)malloc(sizeof(PNM_APPLICATION_T));
  memset(ptPnmApp, 0, sizeof(PNM_APPLICATION_T));
  
  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptPnmApp;

  /* allocate memory for read / write buffers */
  ptPnmApp->tCommon.ulReadBufferSize  = MAX_RTD_SIZE * sizeof(TLR_UINT8);
  ptPnmApp->tCommon.ulWriteBufferSize = MAX_RTD_SIZE * sizeof(TLR_UINT8);

  ptPnmApp->tCommon.pabReadBuffer  =  (TLR_UINT8*) malloc(ptPnmApp->tCommon.ulReadBufferSize);
  ptPnmApp->tCommon.pabWriteBuffer =  (TLR_UINT8*) malloc(ptPnmApp->tCommon.ulWriteBufferSize);

  /* initialize the read and write buffer with zero */
  memset(ptApp->tCommon.pabReadBuffer,  0, ptApp->tCommon.ulReadBufferSize);
  memset(ptApp->tCommon.pabWriteBuffer, 0, ptApp->tCommon.ulWriteBufferSize);

  /* allocate memory for Profinet IO-Devices */
  uiLenMemory = IO_DEVICE_COUNT * sizeof(*ptPnmApp->tPnm.patIoDevices);
  ptPnmApp->tPnm.patIoDevices    = (PNM_IO_DEVICE_PARAMETERS_T*)malloc(uiLenMemory);
  memset(ptPnmApp->tPnm.patIoDevices, 0, uiLenMemory);
  ptPnmApp->tPnm.usIoDeviceCount = IO_DEVICE_COUNT;

  /* allocate memory for submodules */
  for (uiCnt = 0; uiCnt < IO_DEVICE_COUNT; uiCnt++)
  {
    uiLenMemory = MAX_SUBMODULE_COUNT * sizeof(*ptPnmApp->tPnm.patIoDevices->patSubm);
    ptPnmApp->tPnm.patIoDevices[uiCnt].patSubm = (PNM_SUBMODULE_T*)malloc(uiLenMemory);
    memset(ptPnmApp->tPnm.patIoDevices[uiCnt].patSubm, 0, uiLenMemory);
  }

  /* return the pointer */
  *pptApp = ptApp;




  /* add the Profinet parameters to the resources */
  /* this parameters have to be taken from the IO-Device GSDML file and engineering intelligence */

  /* IO-Device is identified by its NameOfStation */
  memcpy(&ptPnmApp->tPnm.patIoDevices[0].abNameOfStation, "cifxrepns-1", strlen("cifxrepns-1"));
  
  /* the IP parameters the IO-Device shall use */
  ptPnmApp->tPnm.patIoDevices[0].ulIpAddr      = 0xC0A8Fc80;  /* 192.168.252.128 */
  ptPnmApp->tPnm.patIoDevices[0].ulNetmask     = 0xFFFFFF00;  /* 255.255.255.0 */
  ptPnmApp->tPnm.patIoDevices[0].ulGateway     = 0;

  /* the ARUUID must be unique between the IO-Controller and each single IO-Device */
  /* it is an Identifier for the Application Relation */
  ptPnmApp->tPnm.patIoDevices[0].ulArUuid1     = 1;
  ptPnmApp->tPnm.patIoDevices[0].usArUuid2     = 1;
  ptPnmApp->tPnm.patIoDevices[0].usArUuid3     = 1;

  /* identification parameters contained in GSDML file */
  ptPnmApp->tPnm.patIoDevices[0].usDeviceId    = 0x0103;
  ptPnmApp->tPnm.patIoDevices[0].usVendorId    = 0x011E;
  ptPnmApp->tPnm.patIoDevices[0].usInstance    = 1;      /* GSDML ObjectUUID_LocalIndex */
  
  /* cycle time definition. The cycle time is 31,25s * SendClock * Reducation */
  ptPnmApp->tPnm.patIoDevices[0].usReductRatio = 16;
  ptPnmApp->tPnm.patIoDevices[0].usSendClock   = 32;

  /* the submodule configuration of the first IO-Device follows */
  /* all parameters are taken from the GSDML file */
  
  /* DAP submodule */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].ulApi          = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].usSlot         = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].usSubslot      = 1;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].ulModuleId     = 0x1001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].ulSubmodId     = 0x1000;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].usDPMOffsIn    = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].usDPMOffsOut   = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].usInputLength  = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].usLenRecord    = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[0].usOutputLength = 0;

  /* PDEV Interface submodule */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].ulApi          = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].usSlot         = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].usSubslot      = 0x8000;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].ulModuleId     = 0x1001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].ulSubmodId     = 0x1001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].usDPMOffsIn    = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].usDPMOffsOut   = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].usInputLength  = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].usLenRecord    = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[1].usOutputLength = 0;

  /* PDEV Port submodule */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].ulApi          = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].usSlot         = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].usSubslot      = 0x8001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].ulModuleId     = 0x1001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].ulSubmodId     = 0x1001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].usDPMOffsIn    = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].usDPMOffsOut   = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].usInputLength  = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].usLenRecord    = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[2].usOutputLength = 0;

  /* PDEV Port submodule */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].ulApi          = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].usSlot         = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].usSubslot      = 0x8002;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].ulModuleId     = 0x1001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].ulSubmodId     = 0x1001;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].usDPMOffsIn    = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].usDPMOffsOut   = 0; /* irrelevant for submodules without IO-data */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].usInputLength  = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].usLenRecord    = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[3].usOutputLength = 0;

  /* 1 Byte Input submodule */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].ulApi          = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].usSlot         = 1;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].usSubslot      = 1;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].ulModuleId     = 2;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].ulSubmodId     = 1;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].usDPMOffsIn    = 0; /* offset in DPM where data of this submodule is copied to */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].usDPMOffsOut   = 0; /* irrelevant for Input submodules */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].usInputLength  = 1;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].usLenRecord    = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[4].usOutputLength = 0;

  /* 1 Byte Output submodule */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].ulApi          = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].usSlot         = 2;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].usSubslot      = 1;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].ulModuleId     = 3;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].ulSubmodId     = 2;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].usDPMOffsIn    = 0; /* irrelevant for Output submodules */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].usDPMOffsOut   = 0; /* offset in DPM where data of this submodule is taken from */
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].usInputLength  = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].usLenRecord    = 0;
  ptPnmApp->tPnm.patIoDevices[0].patSubm[5].usOutputLength = 1;

  /* store the amount of submodules */
  ptPnmApp->tPnm.patIoDevices[0].usSubmodCnt = 6;



  /* start of parameters for the second IO-Device which is only a dummy IO-Device */
  memcpy(&ptPnmApp->tPnm.patIoDevices[1].abNameOfStation, "config-dummy", strlen("config-dummy"));
  ptPnmApp->tPnm.patIoDevices[1].ulArUuid1     = 2;
  ptPnmApp->tPnm.patIoDevices[1].ulGateway     = 0;
  ptPnmApp->tPnm.patIoDevices[1].ulIpAddr      = 0xC0A8Fc81;  /* 192.168.252.129 */
  ptPnmApp->tPnm.patIoDevices[1].ulNetmask     = 0xFFFFFF00;  /* 255.255.255.0 */
  ptPnmApp->tPnm.patIoDevices[1].usArUuid2     = 2;
  ptPnmApp->tPnm.patIoDevices[1].usArUuid3     = 2;
  ptPnmApp->tPnm.patIoDevices[1].usDeviceId    = 0x0103;
  ptPnmApp->tPnm.patIoDevices[1].usVendorId    = 0x011E;
  ptPnmApp->tPnm.patIoDevices[1].usInstance    = 1;
  ptPnmApp->tPnm.patIoDevices[1].usReductRatio = 16;
  ptPnmApp->tPnm.patIoDevices[1].usSendClock   = 32;
  
  /* configure the submodles */
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].ulApi          = 0;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].usSlot         = 0;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].usSubslot      = 1;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].ulModuleId     = 1;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].ulSubmodId     = 1;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].usDPMOffsIn    = 0;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].usDPMOffsOut   = 0;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].usInputLength  = 0;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].usLenRecord    = 0;
  ptPnmApp->tPnm.patIoDevices[1].patSubm[0].usOutputLength = 0;

  ptPnmApp->tPnm.patIoDevices[1].usSubmodCnt   = 1;
  

  DEBUG("Successful.\n");
  return tResult;
}

/*************************************************************************************************
 * @brief This function frees all resources created by App_CreateResources(). 
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_FreeResources(APPLICATION_T* ptApp)
{
  DEBUG("Free resources...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;
  /* temp. counter */
  TLR_UINT   uiCnt;

  /* convert to specific application */
  PNM_APPLICATION_T* ptPnmApp = (PNM_APPLICATION_T*)ptApp;

    /* allocate memory for submodules */
  for (uiCnt = 0; uiCnt < IO_DEVICE_COUNT; uiCnt++)
  {
    if (NULL != ptPnmApp->tPnm.patIoDevices)
    {
      if (NULL != ptPnmApp->tPnm.patIoDevices[uiCnt].patSubm)
      {
        free(ptPnmApp->tPnm.patIoDevices[uiCnt].patSubm);
      }
    }
  }
  
  if (NULL != ptPnmApp->tPnm.patIoDevices)
  {
    free(ptPnmApp->tPnm.patIoDevices);
  }


  /* free buffer resources */
  free(ptPnmApp->tCommon.pabReadBuffer);
  free(ptPnmApp->tCommon.pabWriteBuffer);

  /* free application data container */
  free(ptApp);

  DEBUG("Successful.\n");
  return tResult;
}

/*************************************************************************************************
 * @brief This method handles all packets which will be received by CoE application.
 * 
 * @param ptApp Pointer to application data.
 * @param ptPacket Pointer to CifX packet.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandlePacket(APPLICATION_T*  ptApp, CIFX_PACKET* ptPacket)
{
  TLR_PACKET_HEADER_T* ptPck = (TLR_PACKET_HEADER_T*)ptPacket;

  /* convert to specific application */
  PNM_APPLICATION_T* ptPnmApp = (PNM_APPLICATION_T*)ptApp;

  switch(ptPck->ulCmd)
  {
    default:
      DEBUG("Unknown packet. ulCmd=0x%x\n", ptPacket->tHeader.ulCmd);
      break;
  }
}

/*************************************************************************************************
 * @brief This method handles the cyclic process data exchange.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandleProcessData(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* convert to specific application */
  PNM_APPLICATION_T* ptPnmApp = (PNM_APPLICATION_T*)ptApp;

  /* cyclic io handing happens here */
  /* read RTD-Data */
  tResult = xChannelIORead(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulReadBufferSize, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulReadBufferSize);

  /* the action depends on the return value we have received */
  if ((CIFX_DEV_EXCHANGE_FAILED == tResult) || (CIFX_DEV_EXCHANGE_TIMEOUT == tResult))
  {
    DEBUG("Read failed with 0x%08x\n", tResult);
  }
  else if (CIFX_DEV_NO_COM_FLAG == tResult)
  {
    //DEBUG("Read failed. Device is not communicating\n");
  }
  else if (CIFX_NO_ERROR == tResult)
  {
    //DEBUG("Read succeeded. Copy Inputs back to Outputs.\n");
    
    /* we copy the memory we have read to the memory we want to send, */
    /* because we just want to mirror the data */
    memcpy (ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulWriteBufferSize);     

    // write IO-Data 
    tResult = xChannelIOWrite(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulWriteBufferSize, ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.ulWriteBufferSize);
  }
  else
  {
    /* received unexpected failure */
    DEBUG("Read failed unexpected with 0x%08x\n", tResult);
  }

}

/*************************************************************************************************
 * @brief This function initializes the application. 
 * Objects will be creates and services will be registered inside.
 * App_Finalize() must be called in order to achieve a friendly shutdown of application.
 * 
 * @param ptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Initialize(APPLICATION_T* ptApp)
{
  DEBUG("Initializing application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* convert to specific application */
  PNM_APPLICATION_T* ptPnmApp = (PNM_APPLICATION_T*)ptApp;

  /* register on EcatEsm task (indications for status) */
  tResult = App_SendRecvEmptyPkt(ptApp, RCX_REGISTER_APP_REQ);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }  
  
  return tResult;
}


/*************************************************************************************************
 * @brief This method finalizes the application. 
 * It returns handles, aso.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Finalize(APPLICATION_T* ptApp)
{
  DEBUG("Shutdown application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to specific application */
  PNM_APPLICATION_T* ptPnmApp = (PNM_APPLICATION_T*)ptApp;

  /* at first we empty the queue, than we start with the shutdown sequence below */
  while(TLR_S_OK == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(tRecvPkt), &tRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    App_HandlePacket(ptApp, &tRecvPkt);
  }

  /* unregister on Profinet IO-Controller */
  App_SendRecvEmptyPkt(ptApp, RCX_UNREGISTER_APP_REQ);

  /* perform ChannelInit on Profinet IO-Controller to stop any communication and delete configuration */
  App_SendRecvEmptyPkt(ptApp, RCX_CHANNEL_INIT_REQ);

  return tResult;
}

/*************************************************************************************************
 * @brief This function prompts a debug message at application startup.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_PromptIntro(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  DEBUG("\n");
  DEBUG("**********************************************************\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Configuration example for Profinet IO-Controller     *\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Copyright (c) Hilscher GmbH. All Rights Reserved.    *\n");
  DEBUG("*                                                        *\n");
  DEBUG("**********************************************************\n");
  DEBUG("\n");

  return tResult;
}



